using System.Net;
using System.Net.Sockets;
using System.Text;
using GatewaySCO;
using Serilog;

namespace gatewaySCO.POSBC;

// Maneja conexión al POSBC.
public class ClienteServidorPOSBC(string ip, int pto)
{
    ILogger log = Log.ForContext<ClienteServidorPOSBC>();
    private int _pto = pto;
    private string _ip = ip;
    private Socket _socket = null;
    public bool ConexionActiva { get; private set; } = false;
    private int _contadorMensajesEnviados = 0;
    private int _contadorMensajesRecibidos = 0;

    public void AbreConexion()
    {
        _socket = new(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
        try
        {
            // Conectarse al servidor
            _socket.Connect(new IPEndPoint(IPAddress.Parse(_ip), _pto));
            ConexionActiva = true;
            Log.Information("Conectado a POSBC {ip}:{pto} - {conectado}", _ip, _pto, _socket.Connected);
        }
        catch (SocketException ex)
        {
            ConexionActiva = false;
            Log.Error("Excepción socket conexión a POSBC {ip}:{pto} - código error socket {codigo} - {e}", _ip, ex.SocketErrorCode, _pto, ex);
        }
        catch (Exception e)
        {
            ConexionActiva = false;
            Log.Error("Excepción en conexión a POSBC {ip}:{pto} - {e}", _ip, _pto, e);
        }
    }

    public void CierraConexion()
    {
        try
        {
            // Cerrar el socket
            ConexionActiva = false;
            _socket.Shutdown(SocketShutdown.Both);
            _socket.Close();
        }
        catch (Exception e)
        {
            Log.Warning("Excepción cerrando conexión a POSBC {ip}:{pto} - {e}", _ip, _pto, e);
        }
        Log.Information("Conexión cerrada a POSBC {ip}:{pto} - {conectado}", _ip, _pto, _socket.Connected);
    }

    // Envia mensaje, espera respuesta y la retorna como un arreglo de bytes.
    // El mensaje es un string, convertido a arreglo de bytes, y le antepone
    // 4 bytes con la longitud del mensaje.
    // TODO : validar que la respuesta sean mas de 1 mensaje!!!
    public byte[] EnviaRecibe(byte[] bytesMensaje)
    {
        // Remite mensaje.
        _socket.Send(bytesMensaje);
        _contadorMensajesEnviados++;
        Log.Information("Mensaje #{contador} para POSBIC {ip}", _contadorMensajesEnviados, _ip);
        Log.Information("Esperando respuesta..");

        // Leer la respuesta del servidor
        byte[] buffer = new byte[1024]; // Tamaño del buffer inicial
        int totalBytesRead = 0;
        int bytesRead;
        byte[] bufferEntrada;

        // Usar un MemoryStream para manejar datos de longitud arbitraria
        using (var ms = new System.IO.MemoryStream(1024))
        {
            while ((bytesRead = _socket.Receive(buffer)) > 0)
            {
                ms.Write(buffer, 0, bytesRead);
                totalBytesRead += bytesRead;

                // Romper el ciclo si se ha leído todo el mensaje
                if (_socket.Available == 0) break;
            }

            // Obtener todos los datos recibidos como un arreglo de bytes
            bufferEntrada = ms.ToArray();
        }
        _contadorMensajesRecibidos++;

        Log.Information("Respuesta #{contador} de POSBC {ip}", _contadorMensajesRecibidos, _ip);
        Log.Debug("Mensaje - {msj}", _contadorMensajesRecibidos, _ip, Encoding.UTF8.GetString(bufferEntrada));
        Log.Debug(Util.DetalleMensajes(bufferEntrada));
        return bufferEntrada;
    }
}